/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#if MX_OS_WINNT
#include "getopt.h"
#else
#include <unistd.h>
#include <netinet/in.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>

#include "mx__lib_types.h"
#include "mx__lib.h"
#include "mx__driver_interface.h"
#include "mx__fops.h"
#include "mcp_config.h"

#define MAX_LOG	 MX_MCP_VPAGE_SHIFT
#define MAX_SIZE (1 << MAX_LOG)
#define DEFAULT_ITERATIONS 64

void
usage()
{
  fprintf(stderr, "Usage: mx_dmabench [args]\n");
  fprintf(stderr, "-b - Board number [0]\n");
  fprintf(stderr, "-s - DMA size [%d].  Must be a power of two between 1 and %d\n",
	  MAX_SIZE, MAX_SIZE);
  fprintf(stderr, "-r - Test only read (send) DMA\n");
  fprintf(stderr, "-w - Test only write (recv) DMA\n");
  fprintf(stderr, "-i - Number of iterations [%d]\n",
	  DEFAULT_ITERATIONS);
  fprintf(stderr, "-a - Test all (power of 2) sizes\n");
}



void
run_test(mx_endpt_handle_t fd, mx_dmabench_t *x, float cpu_freq)
{
  double useconds, bytes;
  int status;

  status = mx__run_dmabench(fd, x);
  if (status) {
    perror("dma bench failed");
    goto abort_with_fd;
  }

  useconds = (double) x->cycles / cpu_freq;
  bytes = (double) x->count * (double) (1 << x->log_size);
  printf("DMA %s Bandwidth = %7.2f MB/sec (%4d bytes per DMA)\n", 
	 x->dma_read ? "read  (send)" : "write (recv)",
	 (bytes  / useconds) * (1000000.0 / (1024.0 * 1024.0)), 
	 (1 << x->log_size));

  return;
 abort_with_fd:
  mx__close(fd);
  mx_finalize();
  exit(1);
}

static void
report_error(int board, const char *s)
{
  fprintf(stderr, "Error: mx%d:%s\n", board, s);
  exit(1);
}

int 
main(int argc, char **argv)
{
  mx_endpt_handle_t fd;
  mx_return_t ret;
  uint32_t board_id;
  uint32_t xfer;
  extern char *optarg;
  int c;
  int iterations, do_read, do_one_direction, do_all_sizes;
  mx_dmabench_t x;
  int log_size, size, min_log, max_log;
  mx_get_nic_id_t mac;
  float cpu_freq, pci_freq;

  board_id = 0;
  iterations = DEFAULT_ITERATIONS;
  do_read = 0;
  do_one_direction = 0;
  do_all_sizes = 0;
  size = MAX_SIZE;

  while ((c = getopt(argc, argv, "hrwai:b:s:"))!= EOF) switch(c) {
  case 'h':
    usage();
    exit(0);
    break;
  case 'i':
    iterations = atoi(optarg);
    break;
  case 'b':
     board_id = atoi(optarg);
    break;
  case 'r':
    do_read = 1;
    do_one_direction = 1;
    break;
  case 'w':
    do_read = 0;
    do_one_direction = 1;
    break;
  case 'a':
    do_all_sizes = 1;
    break;
  case 's':
    size = atoi(optarg);
    if (size > MAX_SIZE || size <= 0) {
      fprintf(stderr, "size must be a power of 2 between 1 and %d\n",
	      MAX_SIZE);
      exit(1);
    }
    break;
  default:
    usage();
    exit(1);
  }

  for (log_size = -1; size > 0; log_size++) {
    size = size >> 1;
  }
  mx_init();
  ret = mx_open_any_board(&fd);
  if (ret != MX_SUCCESS) {
    printf("open failed: %s\n", mx_strerror(ret));
    return 0;
  }


  mac.board_number = board_id;
  if (mx__get_nic_id(fd, &mac) != 0) {
    perror("Can't determine mac address\n");
    goto abort_with_fd;
  }
  else {
    char buf[18];
    mx_get_board_val_t type;
    uint32_t pci_freq_raw;

    printf("DMA timings for mx%d (%s):\n", board_id, 
	   mx__nic_id_to_str(buf, mac.nic_id, 18));

    type.board_number = board_id;
    if (mx__get_board_type(fd, &type) != 0)
      report_error(board_id, "Cannot get board_type");

    xfer = board_id;
    cpu_freq = pci_freq = 0.0;
    if (mx__get_cpu_freq(fd, &xfer) != 0)
      report_error(board_id, "Cannot get cpu freq");
    cpu_freq = (float)xfer;
    cpu_freq = cpu_freq/1024.0;

    xfer = board_id;
    if (mx__get_pci_freq(fd, &xfer) != 0) 
      report_error(board_id, "Cannot get pci freq");
    pci_freq_raw = xfer;

    if (cpu_freq != 0.0)
      printf("\tLANai: %3.1f MHz", cpu_freq);
    
    if (pci_freq_raw != 0) {
      if (type.val == MX_BOARD_TYPE_Z) {
	printf("\tPCI-E x%d", xfer);
      } else {
	pci_freq = ((float)xfer / 8192.0) * cpu_freq;
	printf("\tPCI bus: %3.0f MHz", pci_freq);
      }
    }
    
    printf("\t(1MB = 1048576 Bytes)\n");
  }
  x.board_number = board_id;
  x.count = iterations;
  x.dma_read = do_read;
  x.log_size = log_size;

  if (do_all_sizes) {
    min_log = 0;
    max_log = MAX_LOG;
  } else {
    min_log = max_log = log_size;
  }
  

  if (!do_one_direction) {
    x.dma_read = 1;
    for (log_size = min_log; log_size <= max_log; log_size++) {
      x.log_size = log_size;
      run_test(fd, &x, cpu_freq);
    }
    x.dma_read = 0;
  }

  for (log_size = min_log; log_size <= max_log; log_size++) {
    x.log_size = log_size;
    run_test(fd, &x, cpu_freq);
  }


 abort_with_fd:
  mx__close(fd);
  mx_finalize();

  return 0;
}

